/** @file keyboard_bind.h

	HITACHI MB-S1 Emulator 'EmuB-S1'
	Skelton for retropc emulator

	@author Sasaji
	@date   2012.04.08 -

	@brief [ keyboard bind ]
*/

#include "../vm.h"
#include "../../keycode.h"
#include "../../osd/keybind.h"

#define VK_KEY_TYPE "BML3"

/* s1        -> common  win(vc++) win(sdl)/linux    mac          */
/* [caps]    -> F6      capslock  shift + capslock  capslock     */
/* [graph]   -> F7      muhenkan  muhenkan          eisu(jp kb)  */
/* [kana]    -> F8      katakana                    rightoption  */
/* [break]   -> F9                                               */
/* [num ?]   -> end                                              */
/* [num ,]   -> pagedown                                         */
/* [henkan]  ->         henkan    henkan            rightcommand */
/* [copy]    -> F12                                              */
/* reset     ->         F11       F11               F10          */
/* (pause)   ->         pause     pause             F15          */
/* (alt)     ->         alt       alt               leftoption   */

static const uint32_key_assign_t scan2key_defmap[KEYBIND_KEYS]={
//
	{KEYCODE_SPACE,  0x00}, // 0x80, space
	{KEYCODE_UP,     0x00}, // 0x81, up
	{KEYCODE_END,    0x00}, // 0x82, num ?
	{KEYCODE_LEFT,   0x00}, // 0x83, left
	{KEYCODE_DOWN,   0x00}, // 0x84, down
	{KEYCODE_RIGHT,  0x00}, // 0x85, right
	{KEYCODE_LCTRL,  KEYCODE_RCTRL},  // 0x86, ctrl
	{KEYCODE_LSHIFT, KEYCODE_RSHIFT}, // 0x87, shift
//
	{0x00,                0x00},        // 0x88,
	{KEYCODE_CAPSLOCK,    KEYCODE_F6},  // 0x89, caps
#if defined(__APPLE__) && defined(__MACH__)
	{KEYCODE_RALT,        KEYCODE_F8},  // 0x8a, kana
	{KEYCODE_EISU,        KEYCODE_F7},  // 0x8b, graph
#elif defined(USE_WIN)
	{KEYCODE_KATAHIRA,    KEYCODE_F8},  // 0x8a, kana
	{KEYCODE_MUHENKAN,    KEYCODE_F7},  // 0x8b, graph
#else
	{KEYCODE_F8,          0x00},        // 0x8a, kana
	{KEYCODE_MUHENKAN,    KEYCODE_F7},  // 0x8b, graph
#endif
	{KEYCODE_ESCAPE,      0x00}, // 0x8c, esc
	{KEYCODE_KP_8,        0x00}, // 0x8d, num8
	{KEYCODE_KP_9,        0x00}, // 0x8e, num9
	{KEYCODE_KP_MULTIPLY, 0x00}, // 0x8f, num*
//
	{KEYCODE_7,     0x00}, // 0x90, 7
	{KEYCODE_4,     0x00}, // 0x91, 4
	{KEYCODE_6,     0x00}, // 0x92, 6
	{KEYCODE_8,     0x00}, // 0x93, 8
	{KEYCODE_0,     0x00}, // 0x94, 0
	{KEYCODE_CARET, 0x00}, // 0x95, ^
	{KEYCODE_MINUS, 0x00}, // 0x96, =^
	{KEYCODE_3,     0x00}, // 0x97, 3
//
	{0x00,              0x00}, // 0x98
	{KEYCODE_5,         0x00}, // 0x99, 5
	{KEYCODE_1,         0x00}, // 0x9a, 1
	{KEYCODE_2,         0x00}, // 0x9b, 2
	{KEYCODE_9,         0x00}, // 0x9c, 9
	{KEYCODE_KP_7,      0x00}, // 0x9d, num7
	{KEYCODE_BACKSPACE, 0x00}, // 0x9e, bs
	{KEYCODE_BACKSLASH, 0x00}, // 0x9f, yen(backslash)
//
	{KEYCODE_U,        0x00}, // 0xa0, U
	{KEYCODE_R,        0x00}, // 0xa1, R
	{KEYCODE_Y,        0x00}, // 0xa2, Y
	{KEYCODE_I,        0x00}, // 0xa3, I
	{KEYCODE_P,        0x00}, // 0xa4, P
	{KEYCODE_LBRACKET, 0x00}, // 0xa5, [
	{KEYCODE_AT,       0x00}, // 0xa6, @
	{KEYCODE_KP_0,     0x00}, // 0xa7, num0
//
	{KEYCODE_Q,         0x00}, // 0xa8, Q
	{KEYCODE_T,         0x00}, // 0xa9, T
	{KEYCODE_W,         0x00}, // 0xaa, W
	{KEYCODE_E,         0x00}, // 0xab, E
	{KEYCODE_O,         0x00}, // 0xac, O
	{KEYCODE_KP_PERIOD, 0x00}, // 0xad, num.
	{KEYCODE_HOME,      0x00}, // 0xae, home/cls
	{KEYCODE_KP_ENTER,  KEYCODE_RETURN}, // 0xaf, return
//
	{KEYCODE_J,         0x00}, // 0xb0, J
	{KEYCODE_F,         0x00}, // 0xb1, F
	{KEYCODE_H,         0x00}, // 0xb2, H
	{KEYCODE_K,         0x00}, // 0xb3, K
	{KEYCODE_SEMICOLON, 0x00}, // 0xb4, +;
	{KEYCODE_RBRACKET,  0x00}, // 0xb5, ]
	{KEYCODE_COLON,     0x00}, // 0xb6, *:
	{KEYCODE_KP_4,      0x00}, // 0xb7, num4
//
	{KEYCODE_A,        0x00}, // 0xb8, A
	{KEYCODE_G,        0x00}, // 0xb9, G
	{KEYCODE_S,        0x00}, // 0xba, S
	{KEYCODE_D,        0x00}, // 0xbb, D
	{KEYCODE_L,        0x00}, // 0xbc, L
	{KEYCODE_KP_5,     0x00}, // 0xbd, num5
	{KEYCODE_KP_6,     0x00}, // 0xbe, num6
	{KEYCODE_KP_MINUS, 0x00}, // 0xbf, num-
//
	{KEYCODE_M,          0x00}, // 0xc0, M
	{KEYCODE_V,          0x00}, // 0xc1, V
	{KEYCODE_N,          0x00}, // 0xc2, N
	{KEYCODE_COMMA,      0x00}, // 0xc3, <,
	{KEYCODE_SLASH,      0x00}, // 0xc4, ?/
	{KEYCODE_KP_DIVIDE,  0x00}, // 0xc5, num/
	{KEYCODE_UNDERSCORE, 0x00}, // 0xc6, _
	{KEYCODE_KP_1,       0x00}, // 0xc7, num1
//
	{KEYCODE_Z,       0x00}, // 0xc8, Z
	{KEYCODE_B,       0x00}, // 0xc9, B
	{KEYCODE_X,       0x00}, // 0xca, X
	{KEYCODE_C,       0x00}, // 0xcb, C
	{KEYCODE_PERIOD,  0x00}, // 0xcc, >.
	{KEYCODE_KP_2,    0x00}, // 0xcd, num2
	{KEYCODE_KP_3,    0x00}, // 0xce, num3
	{KEYCODE_KP_PLUS, 0x00}, // 0xcf, num+
//
	{KEYCODE_F1, 0x00}, // 0xd0, F1
	{KEYCODE_F2, 0x00}, // 0xd1, F2
	{KEYCODE_F3, 0x00}, // 0xd2, F3
	{KEYCODE_F4, 0x00}, // 0xd3, F4
	{KEYCODE_F5, 0x00}, // 0xd4, F5
	{0x00,       0x00}, // 0xd5,
	{0x00,       0x00}, // 0xd6,
	{0x00,       0x00}, // 0xd7,
//
	{0x00,0x00}, // 0xd8
	{0x00,0x00}, // 0xd9
	{0x00,0x00}, // 0xda
	{0x00,0x00}, // 0xdb
	{0x00,0x00}, // 0xdc
	{0x00,0x00}, // 0xdd
	{0x00,0x00}, // 0xde
	{0x00,0x00}, // 0xdf
//
	{0x00,0x00}, // 0xe0
	{0x00,0x00}, // 0xe1
	{0x00,0x00}, // 0xe2
	{0x00,0x00}, // 0xe3
	{0x00,0x00}, // 0xe4
#if defined(__APPLE__) && defined(__MACH__)
	{KEYCODE_INSERT,   KEYCODE_FUNCTION}, // 0xe5, ins
	{0x00,0x00}, // 0xe6
	{KEYCODE_RGUI,     0x00}, // 0xe7, henkan
#else
	{KEYCODE_INSERT,   0x00}, // 0xe5, ins
	{0x00,0x00}, // 0xe6
	{KEYCODE_HENKAN,   0x00}, // 0xe7, henkan
#endif
//
#if defined(__APPLE__) && defined(__MACH__)
	{KEYCODE_F13,      0x00}, // 0xe8, copy
#else
	{KEYCODE_F12,      0x00}, // 0xe8, copy
#endif
	{0x00,0x00}, // 0xe9
	{0x00,0x00}, // 0xea
	{0x00,0x00}, // 0xeb
	{KEYCODE_TAB,      0x00}, // 0xec, tab
	{KEYCODE_PAGEDOWN, KEYCODE_KP_COMMA}, // 0xed, num,
	{0x00,0x00}, // 0xee
	{KEYCODE_DELETE,   0x00}, // 0xef, del
//
	{0x00,0x00}, // 0xf0
	{0x00,0x00}, // 0xf1
	{0x00,0x00}, // 0xf2
	{0x00,0x00}, // 0xf3
	{0x00,0x00}, // 0xf4
	{0x00,0x00}, // 0xf5
	{0x00,0x00}, // 0xf6
	{0x00,0x00}, // 0xf7
//
#if defined(__APPLE__) && defined(__MACH__)
	{KEYCODE_F16,   0x00},         // (pause)
	{KEYCODE_LALT,  0x00},         // (alt)
#else
	{KEYCODE_PAUSE, 0x00},         // (pause)
	{KEYCODE_LALT,  KEYCODE_RALT}, // (alt)
#endif
	{0x00,0x00},
	{0x00,0x00},
	{0x00,0x00},
	{0x00,0x00},
	{0x00,0x00},
#if defined(__APPLE__) && defined(__MACH__)
	{KEYCODE_F10,   0x00}, // reset
#else
	{KEYCODE_F11,   0x00}, // reset
#endif
	//
	{KEYCODE_F9,    0x00}, // break
	{0x00,0x00}            // (property)
};

/// joypad button -> key scancode assign
static const uint32_key_assign_t joy2key_defmap[KEYBIND_KEYS]={
//	 space                up                num?        left                down                right                ctrl        shift
//   0x80,                0x81,             0x82,       0x83,               0x84,               0x85,                0x86,       0x87,
	{JOYCODE_BTN_A,0x00},{JOYCODE_UP,0x00},{0x00,0x00},{JOYCODE_LEFT,0x00},{JOYCODE_DOWN,0x00},{JOYCODE_RIGHT,0x00},{0x00,0x00},{0x00,0x00},
//	             caps        kana        graph       esc         num8              num9                   num*
//   0x88,       0x89,       0x8a,       0x8b,       0x8c,       0x8d,             0x8e,                  0x8f,
	{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,JOYCODE_UP},{0x00,JOYCODE_UPRIGHT},{0x00,0x00},
//	 7           4           6           8           0           ^           =-          3
//   0x90,       0x91,       0x92,       0x93,       0x94,       0x95,       0x96,       0x97
	{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},
//	             5           1           2           9           num7                  bs          yen(backquote)
//   0x98,       0x99,       0x9a,       0x9b,       0x9c,       0x9d,                 0x9e,       0x9f,
	{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,JOYCODE_UPLEFT},{0x00,0x00},{0x00,0x00},
//	 U           R           Y           I           P           [           @           num0
//   0xa0,       0xa1,       0xa2,       0xa3,       0xa4,       0xa5,       0xa6,       0xa7,
	{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,JOYCODE_BTN_A},
//	 Q           T           W           E           O           num .       home/cls    return
//   0xa8,       0xa9,       0xaa,       0xab,       0xac,       0xad,       0xae,       0xaf,
	{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},
//   J           F           H           K           +;          ]           *:          num4
//   0xb0,       0xb1,       0xb2,       0xb3,       0xb4,       0xb5,       0xb6,       0xb7,
	{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,JOYCODE_LEFT},
//   A           G           S           D           L           num5        num6        num-
//   0xb8,       0xb9,       0xba,       0xbb,       0xbc,       0xbd,       0xbe,       0xbf,
	{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x08},{0x00,0x00},
//   M           V           N           <,          ?/          num/        _           num1
//   0xc0,       0xc1,       0xc2,       0xc3,       0xc4,       0xc5,       0xc6,       0xc7,
	{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,JOYCODE_DOWNLEFT},
//   Z           B           X           C           >.          num2                num3                     num+
//   0xc8,       0xc9,       0xca,       0xcb,       0xcc,       0xcd,               0xce,                    0xcf,
	{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,JOYCODE_DOWN},{0x00,JOYCODE_DOWNRIGHT},{0x00,0x00},
//   F1          F2          F3          F4          F5
//   0xd0,       0xd1,       0xd2,       0xd3,       0xd4,       0xd5,       0xd6,       0xd7,
	{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},
//   0xd8,       0xd9,       0xda,       0xdb,       0xdc,       0xdd,       0xde,       0xdf,
	{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},
//                                                               ins                     henkan
//   0xe0,       0xe1,       0xe2,       0xe3,       0xe4,       0xe5,       0xe6,       0xe7,
	{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},
//   copy                                            tab         num ,                   del
//   0xe8,       0xe9,       0xea,       0xeb,       0xec,       0xed,       0xee,       0xef,
	{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},
//   0xf0,       0xf1,       0xf2,       0xf3,       0xf4,       0xf5,       0xf6,       0xf7,
	{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},
//  (pause)      (alt)                                                                   reset
	{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},
//   break      (property)
	{0x00,0x00},{0x00,0x00}
};

/// joypad button -> PIA joystick assign (S1)
static const uint32_key_assign_t sjoy2joy_defmap[KEYBIND_JOYS] = {
// 0x00- 0x07
	{JOYCODE_UP,		JOYCODE_UP},		// 0x00 up
	{JOYCODE_DOWN,		JOYCODE_DOWN},		// 0x01 down
	{JOYCODE_LEFT,		JOYCODE_LEFT},		// 0x02 left
	{JOYCODE_RIGHT,		JOYCODE_RIGHT},		// 0x03 right
	{JOYCODE_BTN_A,		JOYCODE_BTN_A},		// 0x04 button A
	{0x00,				0x00},				// 0x05 button B
	{0x00,				0x00},
	{0x00,				0x00},
// 0x08 - 0x0f
	{0x00,				0x00},
	{0x00,				0x00},
	{0x00,				0x00},
	{0x00,				0x00},
	{0x00,				0x00},
	{0x00,				0x00},
	{0x00,				0x00},
	{0x00,				0x00},
// 0x10 - 0x17
	{0x00,				0x00},				// 0x10
	{0x00,				0x00},
	{0x00,				0x00},
	{0x00,				0x00},
	{0x00,				0x00},
	{0x00,				0x00},
	{0x00,				0x00},
	{0x00,				0x00},
};

/// joypad button -> PIA joystick assign (SKIPPER)
static const uint32_key_assign_t sjoy2joybit_defmap[KEYBIND_JOYS] = {
// 0x00- 0x07
	{JOYCODE_RIGHT,		0x00}, //   bit 0
	{JOYCODE_LEFT,		0x00}, //   bit 1
	{JOYCODE_UP,		0x00}, //   bit 2
	{JOYCODE_DOWN,		0x00}, //   bit 3
	{JOYCODE_BTN_A,		0x00}, //   bit 4
	{JOYCODE_BTN_B,		0x00}, //   bit 5
	{JOYCODE_BTN_C,		0x00}, //   bit 6
	{0x00,				0x00}, //   bit 7
//   0x08,       0x09,       0x0a,       0x0b,       0x0c,       0x0d,       0x0e,       0x0f,
	{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},
//                                                                                      (property)
//   0x10,       0x11,       0x12,       0x13,       0x14,       0x15,       0x16,       0x17,
	{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},
};

#ifdef USE_PSGJOYSTICK
# ifdef USE_JOYSTICKBIT
/// joypad button -> PSG joystick assign
static const uint32_key_assign_t sjoy2psgjoybit_defmap[KEYBIND_JOYS] = {
// 0x00- 0x07
	{JOYCODE_UP,		0x00}, //   bit 0
	{JOYCODE_DOWN,		0x00}, //   bit 1
	{JOYCODE_RIGHT,		0x00}, //   bit 2
	{JOYCODE_LEFT,		0x00}, //   bit 3
	{JOYCODE_BTN_A,		0x00}, //   bit 4
	{JOYCODE_BTN_B,		0x00}, //   bit 5
	{0x00,				0x00}, //   bit 6
	{0x00,				0x00}, //   bit 7
//   0x08,       0x09,       0x0a,       0x0b,       0x0c,       0x0d,       0x0e,       0x0f,
	{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},
//                                                                                      (property)
//   0x10,       0x11,       0x12,       0x13,       0x14,       0x15,       0x16,       0x17,
	{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},
};
# endif
#endif

/// keyboard -> PIA joystick assign
static const uint32_key_assign_t scan2joy_defmap[KEYBIND_JOYS] = {
// 0x00- 0x07
	{0x00,				0x00},				// 0x00
	{KEYCODE_KP_8,		KEYCODE_UP},		// 0x01 up
	{KEYCODE_KP_2,		KEYCODE_DOWN},		// 0x02 down
	{0x00,				0x00},				// 0x03
	{KEYCODE_KP_4,		KEYCODE_LEFT},		// 0x04 left
	{KEYCODE_KP_7,		0x00},				// 0x05 up+left
	{KEYCODE_KP_1,		0x00},				// 0x06 down+left
	{0x00,				0x00},				// 0x07
// 0x08 - 0x0f
	{KEYCODE_KP_6,		KEYCODE_RIGHT},		// 0x08 right
	{KEYCODE_KP_9,		0x00},				// 0x09 up+right
	{KEYCODE_KP_3,		0x00},				// 0x0A down+right
	{0x00,				0x00},
	{KEYCODE_SPACE,		KEYCODE_N},			// 0x0C button A
	{0x00,				0x00},				// 0x0D button B
	{0x00,				0x00},
	{0x00,				0x00},
// 0x10 - 0x17
	{0x00,				0x00},				// 0x10
	{0x00,				0x00},
	{0x00,				0x00},
	{0x00,				0x00},
	{0x00,				0x00},
	{0x00,				0x00},
	{0x00,				0x00},
	{0x00,				0x00},
};

/// keyboard -> PIA joystick assign (SKIPPER)
static const uint32_key_assign_t scan2joybit_defmap[KEYBIND_JOYS] = {
// 0x00- 0x07
	{KEYCODE_KP_6,		0x00}, //   bit 0
	{KEYCODE_KP_4,		0x00}, //   bit 1
	{KEYCODE_KP_8,		0x00}, //   bit 2
	{KEYCODE_KP_2,		0x00}, //   bit 3
	{KEYCODE_Z,			0x00}, //   bit 4
	{KEYCODE_X,			0x00}, //   bit 5
	{KEYCODE_C,			0x00}, //   bit 6
	{0x00,				0x00}, //   bit 7
//   0x08,       0x09,       0x0a,       0x0b,       0x0c,       0x0d,       0x0e,       0x0f,
	{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},
//                                                                                      (property)
//   0x10,       0x11,       0x12,       0x13,       0x14,       0x15,       0x16,       0x17,
	{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},
};

#ifdef USE_KEY2PSGJOYSTICK
# ifdef USE_JOYSTICKBIT
/// keyboard -> PSG joystick assign
static const uint32_key_assign_t scan2psgjoybit_defmap[KEYBIND_JOYS] = {
// 0x00- 0x07
	{KEYCODE_KP_8,		0x00}, //   bit 0
	{KEYCODE_KP_2,		0x00}, //   bit 1
	{KEYCODE_KP_6,		0x00}, //   bit 2
	{KEYCODE_KP_4,		0x00}, //   bit 3
	{KEYCODE_SPACE,		0x00}, //   bit 4
	{KEYCODE_LSHIFT,	0x00}, //   bit 5
	{0x00,				0x00}, //   bit 6
	{0x00,				0x00}, //   bit 7
//   0x08,       0x09,       0x0a,       0x0b,       0x0c,       0x0d,       0x0e,       0x0f,
	{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},
//                                                                                      (property)
//   0x10,       0x11,       0x12,       0x13,       0x14,       0x15,       0x16,       0x17,
	{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},{0x00,0x00},
};
# endif
#endif

/// for keybind dialog box
///
/// @note index is number in sVmKeyLabels[] on gui_keybinddata.cpp
/// @note Value of each items means the scancode on VM
static const uint16_t kb_scan2key_map[128] = {
//  0    1    2    3    4    5    6    7    8    9    a    b    c    d    e    f
	0xff,0x80,0x7f,0xff,0x78,0x79,0x65,0x6f,0x1e,0x6c,0xff,0xff,0x2e,0x2f,0xff,0xff,
	0x07,0x06,0xff,0xff,0x09,0x0a,0x0b,0x67,0xff,0xff,0xff,0x0c,0x05,0x03,0x01,0x04,
	0x00,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x68,0x6d,
	0x14,0x1a,0x1b,0x17,0x11,0x19,0x12,0x10,0x13,0x1c,0x36,0x34,0x43,0x16,0x4c,0x44,
	0x26,0x38,0x49,0x4b,0x3b,0x2b,0x31,0x39,0x32,0x23,0x30,0x33,0x3c,0x40,0x42,0x2c,
	0x24,0x28,0x21,0x3a,0x29,0x20,0x41,0x2a,0x4a,0x22,0x48,0x25,0x1f,0x35,0x15,0x46,
	0x27,0x47,0x4d,0x4e,0x37,0x3d,0x3e,0x1d,0x0d,0x0e,0x0f,0x4f,0x02,0x3f,0x2d,0x45,
	0x50,0x51,0x52,0x53,0x54,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff
};

/// @note index is number in enVmJoyLabels using cVmJoyLabels[]
static const uint16_t kb_sjoy2joy_map[] = {
//  0    1    2    3    4    5    6    7    8    9    a    b    c    d    e    f
	0xff,0x00,0xff,0x03,0xff,0x01,0xff,0x02,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0x04,0x05
};

/// @note index is number in enVmJoyLabels using cVmJoyLabels[]
static const uint16_t kb_sjoy2joybit_map[] = {
//  0    1    2    3    4    5    6    7    8    9    a    b    c    d    e    f
	0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff
};

/// @note index is number in enVmJoyLabels using cVmJoyLabels[]
static const uint16_t kb_scan2joy_map[] = {
//  0     1     2     3     4     5     6     7     8     9     a     b     c     d     e     f
	0xff, 0x01, 0x09, 0x08, 0x0a, 0x02, 0x06, 0x04, 0x05, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0x10c,0x10d
};
